package architectureRules;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import junit.framework.TestCase;
import thirdPartyToolFramework.AbstractStateListener;
import thirdPartyToolFramework.AbstractUI;
import thirdPartyToolFramework.AbstractUserStateView;
import thirdPartyToolFramework.StateListener;
import thirdPartyToolFramework.UI;
import thirdPartyToolFramework.UserView;
import userFramework.AbstractUserState;
import userFramework.UserState;

public class NotifyEventTest extends TestCase implements UserState {
	private NotifyEventTest servant;
	private StateListener listener;
	private static long MAX_TIME = 100;
	private static int MAX_USERS = 150;
	
	protected void setUp(){

	}
	protected void tearDown(){
		
	}
	public void testLocationTransparency(){
		//the architecture must allow replacement of user
		//states to be transparent to the listeners
		
		//create two versions of the same service
		NotifyEventTest servantA = new NotifyEventTest();
		NotifyEventTest servantB = new NotifyEventTest();
		
		//use the test as a proxy and configure it with a servant. 
		//use the proxy to query data and poll for data
		servant = servantA;
		String firstPoll = this.poll(listener);
		String firstQuery = this.query(UserState.DEFAULT_EVENT);
		
		//run the same test with a different servant
		//and compare the results
		servant = servantB;
		String secondPoll = this.poll(listener);
		String secondQuery = this.query(UserState.DEFAULT_EVENT);
		
		assertTrue(firstPoll.equals(secondPoll));
		assertTrue(firstQuery.equals(secondQuery));
	}
	
	public void testPerformance(){
		//event notifications have to be fast.
		//measure the time from when an event occurs until
		//the notification is received by a subscriber and
		//make sure that the time elapse is acceptable
		List<StateListener> listeners = new ArrayList<StateListener>();
		
		UserState userState = new AbstractUserState();
		listeners.add(newRegisteredListener(userState));
		
		long time = timeToUpdateAndNotifyEvent(userState, listeners);
		assertTrue(time < MAX_TIME);
	}
	public void testScalability(){
		//event notification must scale to 100-150 users
		//the performance of the notification design should
		//scale linearly with the number of users up to the 
		//maximum expected capacity.
		
		List<StateListener> listeners = new ArrayList<StateListener>();
		UserState userState = new AbstractUserState();
		
		for (int i=0; i < MAX_USERS; i++){
			listeners.add(newRegisteredListener(userState));
		}
		long time = timeToUpdateAndNotifyEvent(userState, listeners);
		assertTrue(time < MAX_TIME * MAX_USERS);
	}
	private long timeToUpdateAndNotifyEvent(
			UserState userState, List<StateListener> listeners){
		
		//register all listeners with the given state
		for (StateListener listener : listeners) 
			registerListener(userState, listener);
		
		//update the user state and measure how long it takes
		//the listeners to get an update
		return timeToUpdate(userState, listeners);
	}
	private StateListener newRegisteredListener(UserState userState){
		UI userInterface = new AbstractUI();
		UserView userView = new AbstractUserStateView();
		AbstractStateListener listener = 
			new AbstractStateListener(userState, userInterface, userView);
		registerListener(userState, listener);
		
		return listener;
	}
	private void registerListener(UserState userState, StateListener listener){
		int[] events = {UserState.DEFAULT_EVENT};
		userState.register(listener, events);
	}
	private long timeToUpdate(UserState userState, List<StateListener> listeners){
		
		long startTime = System.currentTimeMillis();
		userState.update(UserState.DEFAULT_EVENT, "Test Event");
		for (StateListener listener : listeners) userState.poll(listener);
		long endTime = System.currentTimeMillis();
		
		return endTime - startTime;
	}
	public String poll(StateListener listener){
		return "Poll Successful";
	}
	public String query(int eventType){
		return "Query Successful";
	}
	public String proxyPoll(StateListener listener){
		return servant.poll(listener);
	}
	public String proxyQuery(int eventType){
		return servant.query(eventType);
	}
	public void register(StateListener listener, int[] events){}
	public void update(int eventType, String event){}
}
